<?php
/**
 * Plugin Name: SEO Buddy
 * Description: One-click SEO diagnostic audit powered by AI
 * Version: 1.2.0
 * Author: SEO Buddy
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: seobuddy
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class SEOBuddy {

	const API_URL = 'https://seobuddy-production.up.railway.app';

	private static $instance = null;

	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	private function __construct() {
		add_action( 'admin_menu', array( $this, 'register_menus' ) );
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_assets' ) );
		add_action( 'wp_ajax_seobuddy_validate_key', array( $this, 'ajax_validate_key' ) );
		add_action( 'wp_ajax_seobuddy_run_audit', array( $this, 'ajax_run_audit' ) );
		add_action( 'wp_ajax_seobuddy_poll_audit', array( $this, 'ajax_poll_audit' ) );
		add_action( 'rest_api_init', array( $this, 'register_mcp_routes' ) );
		add_action( 'enqueue_block_editor_assets', array( $this, 'enqueue_editor_assets' ) );

		// Posts list table column.
		add_filter( 'manage_posts_columns', array( $this, 'add_seo_column' ) );
		add_action( 'manage_posts_custom_column', array( $this, 'render_seo_column' ), 10, 2 );
		add_action( 'admin_head-edit.php', array( $this, 'seo_column_css' ) );

		if ( get_option( 'seobuddy_security_headers', false ) ) {
			add_action( 'send_headers', array( $this, 'send_security_headers' ) );
		}

		// Inject a real text H1 on the homepage (theme puts logo in H1).
		if ( ! is_admin() ) {
			add_action( 'wp_head', array( $this, 'inject_homepage_h1' ) );
			add_action( 'wp_head', array( $this, 'inject_author_schema' ) );
		}
	}

	/**
	 * Inject a screen-reader-accessible H1 on the homepage for SEO.
	 */
	public function inject_homepage_h1() {
		if ( ! is_front_page() ) {
			return;
		}
		$h1_text = esc_html( get_option( 'seobuddy_homepage_h1', 'Pizza Recipes, Facts & Reviews' ) );
		echo '<style>.seobuddy-sr-h1{position:absolute;width:1px;height:1px;padding:0;margin:-1px;overflow:hidden;clip:rect(0,0,0,0);white-space:nowrap;border:0}</style>';
		echo '<h1 class="seobuddy-sr-h1">' . $h1_text . '</h1>';
	}

	/**
	 * Output Person (author) schema on single posts for E-E-A-T.
	 * Outputs on homepage and single posts/pages with an author.
	 */
	public function inject_author_schema() {
		if ( is_front_page() ) {
			// Site-wide author/publisher on homepage only.
			$schema = array(
				'@context' => 'https://schema.org',
				'@type'    => 'Person',
				'@id'      => 'https://pizzaware.com/#author',
				'name'     => 'Pizzaware Editorial Team',
				'url'      => 'https://pizzaware.com/about/',
				'worksFor' => array(
					'@id' => 'https://pizzaware.com/#organization',
				),
				'knowsAbout' => array( 'Pizza', 'Italian Cuisine', 'Pizza Recipes', 'Pizza Reviews' ),
			);
			echo '<script type="application/ld+json">' . wp_json_encode( $schema, JSON_UNESCAPED_SLASHES ) . '</script>' . "\n";
			return;
		}

		if ( ! is_singular( array( 'post', 'page' ) ) ) {
			return;
		}

		$post   = get_queried_object();
		$author = get_the_author_meta( 'display_name', $post->post_author );
		if ( empty( $author ) ) {
			return;
		}

		$schema = array(
			'@context'    => 'https://schema.org',
			'@type'       => 'Article',
			'headline'    => get_the_title( $post ),
			'datePublished' => get_the_date( 'c', $post ),
			'dateModified'  => get_the_modified_date( 'c', $post ),
			'author'      => array(
				'@type' => 'Person',
				'@id'   => 'https://pizzaware.com/#author',
				'name'  => $author,
				'url'   => 'https://pizzaware.com/about/',
			),
			'publisher'   => array(
				'@id' => 'https://pizzaware.com/#organization',
			),
			'mainEntityOfPage' => get_permalink( $post ),
		);
		echo '<script type="application/ld+json">' . wp_json_encode( $schema, JSON_UNESCAPED_SLASHES ) . '</script>' . "\n";
	}

	public function send_security_headers() {
		header( 'Strict-Transport-Security: max-age=31536000; includeSubDomains; preload' );
		header( 'X-Content-Type-Options: nosniff' );
		header( 'X-Frame-Options: SAMEORIGIN' );
		header( 'Referrer-Policy: strict-origin-when-cross-origin' );
		header( 'Permissions-Policy: geolocation=(), microphone=(), camera=()' );
	}

	/* ------------------------------------------------------------------ */
	/* MCP Server (Streamable HTTP transport)                             */
	/* ------------------------------------------------------------------ */

	/**
	 * Register the MCP REST route.
	 */
	public function register_mcp_routes() {
		register_rest_route( 'seobuddy/v1', '/mcp', array(
			'methods'             => 'POST',
			'callback'            => array( $this, 'handle_mcp' ),
			'permission_callback' => function ( $request ) {
				// Authenticate via Application Password (Basic Auth).
				return current_user_can( 'manage_options' );
			},
		) );
	}

	/**
	 * Handle an MCP JSON-RPC request.
	 */
	public function handle_mcp( $request ) {
		$body = $request->get_json_params();

		if ( empty( $body ) || ! isset( $body['method'] ) ) {
			return new WP_REST_Response( array(
				'jsonrpc' => '2.0',
				'id'      => $body['id'] ?? null,
				'error'   => array( 'code' => -32600, 'message' => 'Invalid request' ),
			), 200 );
		}

		$method = $body['method'];
		$id     = $body['id'] ?? null;
		$params = $body['params'] ?? array();

		switch ( $method ) {
			case 'initialize':
				return $this->mcp_initialize( $id, $params );
			case 'tools/list':
				return $this->mcp_tools_list( $id );
			case 'tools/call':
				return $this->mcp_tools_call( $id, $params );
			case 'notifications/initialized':
				// Client notification, no response needed.
				return new WP_REST_Response( '', 202 );
			default:
				return new WP_REST_Response( array(
					'jsonrpc' => '2.0',
					'id'      => $id,
					'error'   => array( 'code' => -32601, 'message' => 'Method not found' ),
				), 200 );
		}
	}

	private function mcp_initialize( $id, $params ) {
		return new WP_REST_Response( array(
			'jsonrpc' => '2.0',
			'id'      => $id,
			'result'  => array(
				'protocolVersion' => '2025-03-26',
				'capabilities'    => array(
					'tools' => new stdClass(),
				),
				'serverInfo'      => array(
					'name'    => 'seobuddy-wp',
					'version' => '1.0.0',
				),
			),
		), 200 );
	}

	private function mcp_tools_list( $id ) {
		$tools = array(
			array(
				'name'        => 'get_posts',
				'description' => 'List WordPress posts/pages. Returns id, title, slug, status, type, and excerpt.',
				'inputSchema' => array(
					'type'       => 'object',
					'properties' => array(
						'post_type' => array(
							'type'        => 'string',
							'description' => 'Post type: post or page',
							'default'     => 'post',
						),
						'per_page' => array(
							'type'        => 'integer',
							'description' => 'Number of results (max 50)',
							'default'     => 20,
						),
						'search' => array(
							'type'        => 'string',
							'description' => 'Search term to filter posts',
						),
					),
				),
			),
			array(
				'name'        => 'get_post',
				'description' => 'Get a single post/page by ID. Returns full content, title, slug, meta, and SEO fields.',
				'inputSchema' => array(
					'type'       => 'object',
					'properties' => array(
						'post_id' => array(
							'type'        => 'integer',
							'description' => 'The post or page ID',
						),
					),
					'required'   => array( 'post_id' ),
				),
			),
			array(
				'name'        => 'update_post',
				'description' => 'Update a post/page. Can change title, content, excerpt, slug, or status.',
				'inputSchema' => array(
					'type'       => 'object',
					'properties' => array(
						'post_id'  => array(
							'type'        => 'integer',
							'description' => 'The post or page ID to update',
						),
						'title'    => array(
							'type'        => 'string',
							'description' => 'New post title',
						),
						'content'  => array(
							'type'        => 'string',
							'description' => 'New post content (HTML)',
						),
						'excerpt'  => array(
							'type'        => 'string',
							'description' => 'New post excerpt',
						),
					),
					'required'   => array( 'post_id' ),
				),
			),
			array(
				'name'        => 'update_post_meta',
				'description' => 'Update post meta fields. Use for Rank Math SEO title (rank_math_title), meta description (rank_math_description), focus keyword (rank_math_focus_keyword), or any custom field.',
				'inputSchema' => array(
					'type'       => 'object',
					'properties' => array(
						'post_id'    => array(
							'type'        => 'integer',
							'description' => 'The post or page ID',
						),
						'meta_key'   => array(
							'type'        => 'string',
							'description' => 'Meta key to update',
						),
						'meta_value' => array(
							'type'        => 'string',
							'description' => 'New meta value',
						),
					),
					'required'   => array( 'post_id', 'meta_key', 'meta_value' ),
				),
			),
			array(
				'name'        => 'get_media',
				'description' => 'List media items. Returns id, title, alt_text, url, and mime_type.',
				'inputSchema' => array(
					'type'       => 'object',
					'properties' => array(
						'per_page' => array(
							'type'        => 'integer',
							'description' => 'Number of results (max 100)',
							'default'     => 20,
						),
						'search' => array(
							'type'        => 'string',
							'description' => 'Search term to filter media',
						),
						'missing_alt' => array(
							'type'        => 'boolean',
							'description' => 'If true, only return images with empty alt text',
						),
					),
				),
			),
			array(
				'name'        => 'update_media_alt',
				'description' => 'Update the alt text of a media item (image).',
				'inputSchema' => array(
					'type'       => 'object',
					'properties' => array(
						'attachment_id' => array(
							'type'        => 'integer',
							'description' => 'The media attachment ID',
						),
						'alt_text'      => array(
							'type'        => 'string',
							'description' => 'New alt text for the image',
						),
					),
					'required'   => array( 'attachment_id', 'alt_text' ),
				),
			),
			array(
				'name'        => 'add_security_headers',
				'description' => 'Add security headers (HSTS, X-Content-Type-Options, X-Frame-Options, CSP) via a must-use plugin. Safe and reversible.',
				'inputSchema' => array(
					'type'       => 'object',
					'properties' => new stdClass(),
				),
			),
			array(
				'name'        => 'get_site_info',
				'description' => 'Get WordPress site info: name, url, description, theme, active plugins, PHP/WP versions.',
				'inputSchema' => array(
					'type'       => 'object',
					'properties' => new stdClass(),
				),
			),
			array(
				'name'        => 'bulk_optimize_images',
				'description' => 'Trigger Imagify bulk image optimization. Optimizes up to 10 unoptimized images per call. Call repeatedly until all done.',
				'inputSchema' => array(
					'type'       => 'object',
					'properties' => array(
						'batch_size' => array(
							'type'        => 'integer',
							'description' => 'Number of images to optimize (max 10, default 10)',
							'default'     => 10,
						),
					),
				),
			),
		);

		return new WP_REST_Response( array(
			'jsonrpc' => '2.0',
			'id'      => $id,
			'result'  => array( 'tools' => $tools ),
		), 200 );
	}

	private function mcp_tools_call( $id, $params ) {
		$tool_name = $params['name'] ?? '';
		$args      = $params['arguments'] ?? array();

		switch ( $tool_name ) {
			case 'get_posts':
				return $this->mcp_result( $id, $this->tool_get_posts( $args ) );
			case 'get_post':
				return $this->mcp_result( $id, $this->tool_get_post( $args ) );
			case 'update_post':
				return $this->mcp_result( $id, $this->tool_update_post( $args ) );
			case 'update_post_meta':
				return $this->mcp_result( $id, $this->tool_update_post_meta( $args ) );
			case 'get_media':
				return $this->mcp_result( $id, $this->tool_get_media( $args ) );
			case 'update_media_alt':
				return $this->mcp_result( $id, $this->tool_update_media_alt( $args ) );
			case 'add_security_headers':
				return $this->mcp_result( $id, $this->tool_add_security_headers() );
			case 'get_site_info':
				return $this->mcp_result( $id, $this->tool_get_site_info() );
			case 'bulk_optimize_images':
				return $this->mcp_result( $id, $this->tool_bulk_optimize_images( $args ) );
			default:
				return new WP_REST_Response( array(
					'jsonrpc' => '2.0',
					'id'      => $id,
					'error'   => array( 'code' => -32602, 'message' => "Unknown tool: $tool_name" ),
				), 200 );
		}
	}

	private function mcp_result( $id, $text ) {
		return new WP_REST_Response( array(
			'jsonrpc' => '2.0',
			'id'      => $id,
			'result'  => array(
				'content' => array(
					array( 'type' => 'text', 'text' => $text ),
				),
			),
		), 200 );
	}

	/* ---- MCP Tool Implementations ---- */

	private function tool_get_posts( $args ) {
		$post_type = sanitize_text_field( $args['post_type'] ?? 'post' );
		$per_page  = min( intval( $args['per_page'] ?? 20 ), 50 );
		$search    = sanitize_text_field( $args['search'] ?? '' );

		$query_args = array(
			'post_type'      => $post_type,
			'posts_per_page' => $per_page,
			'post_status'    => 'any',
		);
		if ( $search ) {
			$query_args['s'] = $search;
		}

		$posts  = get_posts( $query_args );
		$result = array();
		foreach ( $posts as $p ) {
			$result[] = array(
				'id'      => $p->ID,
				'title'   => $p->post_title,
				'slug'    => $p->post_name,
				'status'  => $p->post_status,
				'type'    => $p->post_type,
				'excerpt' => wp_trim_words( $p->post_excerpt ?: $p->post_content, 30 ),
			);
		}
		return wp_json_encode( $result, JSON_PRETTY_PRINT );
	}

	private function tool_get_post( $args ) {
		$post_id = intval( $args['post_id'] ?? 0 );
		$post    = get_post( $post_id );
		if ( ! $post ) {
			return "Error: Post $post_id not found.";
		}
		$meta = get_post_meta( $post_id );
		$seo  = array(
			'seo_title'       => $meta['rank_math_title'][0] ?? '',
			'seo_description' => $meta['rank_math_description'][0] ?? '',
			'focus_keyword'   => $meta['rank_math_focus_keyword'][0] ?? '',
		);
		return wp_json_encode( array(
			'id'      => $post->ID,
			'title'   => $post->post_title,
			'slug'    => $post->post_name,
			'status'  => $post->post_status,
			'type'    => $post->post_type,
			'content' => $post->post_content,
			'excerpt' => $post->post_excerpt,
			'seo'     => $seo,
		), JSON_PRETTY_PRINT );
	}

	private function tool_update_post( $args ) {
		$post_id = intval( $args['post_id'] ?? 0 );
		$post    = get_post( $post_id );
		if ( ! $post ) {
			return "Error: Post $post_id not found.";
		}
		$update = array( 'ID' => $post_id );
		if ( isset( $args['title'] ) ) {
			$update['post_title'] = sanitize_text_field( $args['title'] );
		}
		if ( isset( $args['content'] ) ) {
			$update['post_content'] = wp_kses_post( $args['content'] );
		}
		if ( isset( $args['excerpt'] ) ) {
			$update['post_excerpt'] = sanitize_textarea_field( $args['excerpt'] );
		}
		$result = wp_update_post( $update, true );
		if ( is_wp_error( $result ) ) {
			return 'Error: ' . $result->get_error_message();
		}
		return "Updated post $post_id successfully.";
	}

	private function tool_update_post_meta( $args ) {
		$post_id    = intval( $args['post_id'] ?? 0 );
		$meta_key   = sanitize_text_field( $args['meta_key'] ?? '' );
		$meta_value = sanitize_text_field( $args['meta_value'] ?? '' );

		if ( ! get_post( $post_id ) ) {
			return "Error: Post $post_id not found.";
		}
		update_post_meta( $post_id, $meta_key, $meta_value );
		return "Updated meta '$meta_key' on post $post_id.";
	}

	private function tool_get_media( $args ) {
		$per_page    = min( intval( $args['per_page'] ?? 20 ), 100 );
		$search      = sanitize_text_field( $args['search'] ?? '' );
		$missing_alt = ! empty( $args['missing_alt'] );

		$query_args = array(
			'post_type'      => 'attachment',
			'post_mime_type' => 'image',
			'posts_per_page' => $missing_alt ? 100 : $per_page,
			'post_status'    => 'inherit',
		);
		if ( $search ) {
			$query_args['s'] = $search;
		}

		$attachments = get_posts( $query_args );
		$result      = array();
		foreach ( $attachments as $att ) {
			$alt = get_post_meta( $att->ID, '_wp_attachment_image_alt', true );
			if ( $missing_alt && ! empty( $alt ) ) {
				continue;
			}
			$result[] = array(
				'id'        => $att->ID,
				'title'     => $att->post_title,
				'alt_text'  => $alt,
				'url'       => wp_get_attachment_url( $att->ID ),
				'mime_type' => $att->post_mime_type,
			);
			if ( count( $result ) >= $per_page ) {
				break;
			}
		}
		return wp_json_encode( $result, JSON_PRETTY_PRINT );
	}

	private function tool_update_media_alt( $args ) {
		$att_id   = intval( $args['attachment_id'] ?? 0 );
		$alt_text = sanitize_text_field( $args['alt_text'] ?? '' );

		if ( ! get_post( $att_id ) ) {
			return "Error: Attachment $att_id not found.";
		}
		update_post_meta( $att_id, '_wp_attachment_image_alt', $alt_text );
		return "Updated alt text on attachment $att_id to: \"$alt_text\"";
	}

	private function tool_add_security_headers() {
		if ( get_option( 'seobuddy_security_headers', false ) ) {
			return 'Security headers already enabled.';
		}

		update_option( 'seobuddy_security_headers', true );
		return 'Security headers enabled. HSTS, X-Content-Type-Options, X-Frame-Options, Referrer-Policy, and Permissions-Policy will be sent on all pages.';
	}

	private function tool_get_site_info() {
		$theme   = wp_get_theme();
		$plugins = get_option( 'active_plugins', array() );
		return wp_json_encode( array(
			'name'           => get_bloginfo( 'name' ),
			'url'            => home_url(),
			'description'    => get_bloginfo( 'description' ),
			'wp_version'     => get_bloginfo( 'version' ),
			'php_version'    => phpversion(),
			'theme'          => $theme->get( 'Name' ) . ' ' . $theme->get( 'Version' ),
			'active_plugins' => $plugins,
			'is_multisite'   => is_multisite(),
		), JSON_PRETTY_PRINT );
	}

	private function tool_bulk_optimize_images( $args ) {
		if ( ! function_exists( 'get_imagify_attachment' ) && ! class_exists( '\Imagify\Optimization\Process\Process' ) ) {
			return 'Error: Imagify plugin is not active or not found.';
		}

		$batch_size = min( intval( $args['batch_size'] ?? 10 ), 10 );

		// Find unoptimized images.
		$query_args = array(
			'post_type'      => 'attachment',
			'post_mime_type' => 'image',
			'posts_per_page' => $batch_size,
			'post_status'    => 'inherit',
			'meta_query'     => array(
				array(
					'key'     => '_imagify_status',
					'compare' => 'NOT EXISTS',
				),
			),
		);

		$attachments = get_posts( $query_args );

		if ( empty( $attachments ) ) {
			return 'All images are already optimized. Nothing to do.';
		}

		$results  = array();
		$success  = 0;
		$failed   = 0;

		foreach ( $attachments as $att ) {
			try {
				if ( function_exists( 'get_imagify_attachment' ) ) {
					$attachment = get_imagify_attachment( 'wp', $att->ID, 'post' );
					$res = $attachment->optimize();
				} elseif ( class_exists( '\Imagify\Optimization\Process\Process' ) ) {
					$process = new \Imagify\Optimization\Process\Process( $att->ID );
					$res = $process->optimize();
				} else {
					$res = new \WP_Error( 'no_method', 'No optimization method available' );
				}

				if ( is_wp_error( $res ) ) {
					$failed++;
					$results[] = $att->ID . ': ' . $res->get_error_message();
				} else {
					$success++;
					$results[] = $att->ID . ': optimized';
				}
			} catch ( \Exception $e ) {
				$failed++;
				$results[] = $att->ID . ': ' . $e->getMessage();
			}
		}

		$total_unoptimized = $this->count_unoptimized_images();

		return wp_json_encode( array(
			'batch_optimized' => $success,
			'batch_failed'    => $failed,
			'remaining'       => $total_unoptimized,
			'details'         => $results,
		), JSON_PRETTY_PRINT );
	}

	private function count_unoptimized_images() {
		$query = new \WP_Query( array(
			'post_type'      => 'attachment',
			'post_mime_type' => 'image',
			'posts_per_page' => 1,
			'post_status'    => 'inherit',
			'fields'         => 'ids',
			'meta_query'     => array(
				array(
					'key'     => '_imagify_status',
					'compare' => 'NOT EXISTS',
				),
			),
		) );
		return $query->found_posts;
	}

	/**
	 * Register admin menu pages.
	 */
	public function register_menus() {
		add_options_page(
			'SEO Buddy',
			'SEO Buddy',
			'manage_options',
			'seobuddy-settings',
			array( $this, 'render_settings_page' )
		);

		add_management_page(
			'SEO Buddy',
			'SEO Buddy',
			'manage_options',
			'seobuddy-audit',
			array( $this, 'render_audit_page' )
		);
	}

	/**
	 * Enqueue CSS and JS on plugin pages only.
	 */
	public function enqueue_assets( $hook ) {
		$plugin_pages = array(
			'settings_page_seobuddy-settings',
			'tools_page_seobuddy-audit',
		);

		if ( ! in_array( $hook, $plugin_pages, true ) ) {
			return;
		}

		wp_enqueue_style(
			'seobuddy-admin',
			plugin_dir_url( __FILE__ ) . 'assets/css/admin.css',
			array(),
			'1.0.0'
		);

		wp_enqueue_script(
			'marked',
			'https://cdn.jsdelivr.net/npm/marked/marked.min.js',
			array(),
			null,
			true
		);

		wp_enqueue_script(
			'seobuddy-admin',
			plugin_dir_url( __FILE__ ) . 'assets/js/admin.js',
			array( 'jquery', 'marked' ),
			'1.0.0',
			true
		);

		wp_localize_script( 'seobuddy-admin', 'seobuddy', array(
			'ajaxurl' => admin_url( 'admin-ajax.php' ),
			'nonce'   => wp_create_nonce( 'seobuddy_nonce' ),
			'siteUrl' => home_url(),
		) );
	}

	/**
	 * Enqueue editor sidebar assets (Gutenberg only).
	 */
	public function enqueue_editor_assets() {
		$asset_file = plugin_dir_path( __FILE__ ) . 'assets/js/editor/index.asset.php';
		if ( ! file_exists( $asset_file ) ) {
			return;
		}
		$asset = include $asset_file;

		wp_enqueue_script(
			'seobuddy-editor',
			plugin_dir_url( __FILE__ ) . 'assets/js/editor/index.js',
			$asset['dependencies'],
			$asset['version'],
			true
		);

		if ( file_exists( plugin_dir_path( __FILE__ ) . 'assets/js/editor/index.css' ) ) {
			wp_enqueue_style(
				'seobuddy-editor',
				plugin_dir_url( __FILE__ ) . 'assets/js/editor/index.css',
				array(),
				$asset['version']
			);
		}

		$post_id  = get_the_ID();
		$seo_meta = array();
		if ( $post_id ) {
			$seo_meta = array(
				'seoTitle'     => get_post_meta( $post_id, 'rank_math_title', true ),
				'metaDesc'     => get_post_meta( $post_id, 'rank_math_description', true ),
				'focusKeyword' => get_post_meta( $post_id, 'rank_math_focus_keyword', true ),
			);
		}

		wp_localize_script( 'seobuddy-editor', 'seobuddyEditor', array(
			'siteUrl'        => home_url(),
			'seoMeta'        => $seo_meta,
			'themeRendersH1' => $this->detect_theme_h1(),
		) );
	}

	/**
	 * Detect whether the active theme renders the post title inside an H1 tag.
	 * Fetches one published post and checks. Result is cached in a transient.
	 */
	private function detect_theme_h1() {
		$cached = get_transient( 'seobuddy_theme_h1' );
		if ( false !== $cached ) {
			return (bool) $cached;
		}

		// Find a published post to test against.
		$posts = get_posts( array(
			'post_type'      => 'post',
			'post_status'    => 'publish',
			'posts_per_page' => 1,
			'fields'         => 'ids',
		) );
		if ( empty( $posts ) ) {
			return false;
		}

		$response = wp_remote_get( get_permalink( $posts[0] ), array( 'timeout' => 5 ) );
		if ( is_wp_error( $response ) ) {
			return false;
		}

		$html   = wp_remote_retrieve_body( $response );
		$result = (bool) preg_match( '/<h1[^>]*class="[^"]*entry-title[^"]*"/i', $html );

		// Cache for 24 hours (re-checks after theme switch).
		set_transient( 'seobuddy_theme_h1', $result ? 1 : 0, DAY_IN_SECONDS );

		return $result;
	}

	/**
	 * Get stored license key.
	 */
	private function get_license_key() {
		return get_option( 'seobuddy_license_key', '' );
	}

	/**
	 * Render the Settings page.
	 */
	public function render_settings_page() {
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}

		$license_key = $this->get_license_key();
		?>
		<div class="wrap seobuddy-wrap">
			<h1><?php echo esc_html( 'SEO Buddy Settings' ); ?></h1>

			<div class="seobuddy-card">
				<h2><?php echo esc_html( 'License Key' ); ?></h2>
				<p class="description">
					<?php echo esc_html( 'Enter your SEO Buddy license key to activate the plugin.' ); ?>
				</p>

				<div class="seobuddy-field-row">
					<input
						type="text"
						id="seobuddy-license-key"
						class="regular-text"
						value="<?php echo esc_attr( $license_key ); ?>"
						placeholder="<?php echo esc_attr( 'Enter your license key' ); ?>"
					/>
					<button type="button" id="seobuddy-validate-btn" class="button seobuddy-btn">
						<?php echo esc_html( 'Validate' ); ?>
					</button>
				</div>

				<div id="seobuddy-key-status" class="seobuddy-status" style="display:none;">
					<span class="seobuddy-status-dot"></span>
					<span class="seobuddy-status-text"></span>
				</div>

				<p class="seobuddy-get-key">
					<?php
					printf(
						/* translators: %s: link to get a license key */
						esc_html__( "Don't have a key? %s", 'seobuddy' ),
						'<a href="https://seobuddy.ai" target="_blank" rel="noopener noreferrer">' . esc_html__( 'Get a key', 'seobuddy' ) . '</a>'
					);
					?>
				</p>
			</div>
		</div>
		<?php
	}

	/**
	 * Render the Audit page.
	 */
	public function render_audit_page() {
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}

		$license_key  = $this->get_license_key();
		$has_key      = ! empty( $license_key );
		$last_audit   = get_option( 'seobuddy_last_audit_id', '' );
		?>
		<div class="wrap seobuddy-wrap">
			<h1><?php echo esc_html( 'SEO Buddy' ); ?></h1>

			<?php if ( ! $has_key ) : ?>
				<div class="notice notice-warning">
					<p>
						<?php
						printf(
							esc_html__( 'Please configure your license key in %s first.', 'seobuddy' ),
							'<a href="' . esc_url( admin_url( 'options-general.php?page=seobuddy-settings' ) ) . '">' . esc_html__( 'Settings', 'seobuddy' ) . '</a>'
						);
						?>
					</p>
				</div>
			<?php endif; ?>

			<div class="seobuddy-card">
				<p class="description">
					<?php
					printf(
						esc_html__( 'Run a full SEO diagnostic audit for %s', 'seobuddy' ),
						'<strong>' . esc_html( home_url() ) . '</strong>'
					);
					?>
				</p>

				<button
					type="button"
					id="seobuddy-run-audit-btn"
					class="button button-primary seobuddy-btn"
					<?php disabled( ! $has_key ); ?>
				>
					<?php echo esc_html( 'Run Full Audit' ); ?>
				</button>

				<div id="seobuddy-progress" class="seobuddy-progress" style="display:none;">
					<div class="seobuddy-spinner"></div>
					<span class="seobuddy-progress-text"><?php echo esc_html( 'Starting audit...' ); ?></span>
				</div>
			</div>

			<div id="seobuddy-report-container" class="seobuddy-card" style="display:none;">
				<div id="seobuddy-score" class="seobuddy-score"></div>
				<div id="seobuddy-report" class="seobuddy-report"></div>
			</div>

			<?php if ( ! empty( $last_audit ) ) : ?>
				<input type="hidden" id="seobuddy-last-audit-id" value="<?php echo esc_attr( $last_audit ); ?>" />
			<?php endif; ?>
		</div>
		<?php
	}

	/**
	 * AJAX: Validate license key.
	 */
	public function ajax_validate_key() {
		check_ajax_referer( 'seobuddy_nonce', '_ajax_nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => 'Unauthorized' ), 403 );
		}

		$license_key = isset( $_POST['license_key'] ) ? sanitize_text_field( wp_unslash( $_POST['license_key'] ) ) : '';

		if ( empty( $license_key ) ) {
			wp_send_json_error( array( 'message' => 'License key is required.' ) );
		}

		$response = wp_remote_get(
			self::API_URL . '/license/validate',
			array(
				'headers' => array(
					'X-License-Key' => $license_key,
				),
				'timeout' => 15,
			)
		);

		if ( is_wp_error( $response ) ) {
			wp_send_json_error( array( 'message' => 'Could not connect to SEO Buddy API.' ) );
		}

		$status_code = wp_remote_retrieve_response_code( $response );
		$body        = json_decode( wp_remote_retrieve_body( $response ), true );

		if ( 200 !== $status_code || empty( $body ) ) {
			wp_send_json_error( array( 'message' => 'Invalid license key.' ) );
		}

		$valid = ! empty( $body['valid'] );
		$email = isset( $body['email'] ) ? sanitize_email( $body['email'] ) : '';

		if ( $valid ) {
			update_option( 'seobuddy_license_key', $license_key );
		}

		wp_send_json_success( array(
			'valid' => $valid,
			'email' => $email,
		) );
	}

	/**
	 * AJAX: Run a new audit.
	 */
	public function ajax_run_audit() {
		check_ajax_referer( 'seobuddy_nonce', '_ajax_nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => 'Unauthorized' ), 403 );
		}

		$license_key = $this->get_license_key();

		if ( empty( $license_key ) ) {
			wp_send_json_error( array( 'message' => 'License key not configured.' ) );
		}

		$site_url = home_url();

		$response = wp_remote_post(
			self::API_URL . '/audits',
			array(
				'headers' => array(
					'X-License-Key' => $license_key,
					'Content-Type'  => 'application/json',
				),
				'body'    => wp_json_encode( array(
					'site_url' => $site_url,
				) ),
				'timeout' => 30,
			)
		);

		if ( is_wp_error( $response ) ) {
			wp_send_json_error( array( 'message' => 'Could not connect to SEO Buddy API.' ) );
		}

		$status_code = wp_remote_retrieve_response_code( $response );
		$body        = json_decode( wp_remote_retrieve_body( $response ), true );

		if ( $status_code < 200 || $status_code >= 300 || empty( $body ) ) {
			$error_message = isset( $body['error'] ) ? $body['error'] : 'Failed to start audit.';
			wp_send_json_error( array( 'message' => $error_message ) );
		}

		wp_send_json_success( array(
			'audit_id' => isset( $body['audit_id'] ) ? sanitize_text_field( $body['audit_id'] ) : '',
			'status'   => isset( $body['status'] ) ? sanitize_text_field( $body['status'] ) : '',
		) );
	}

	/* ------------------------------------------------------------------ */
	/* Posts list table column                                             */
	/* ------------------------------------------------------------------ */

	/**
	 * Add the SEO Buddy column to the posts list table.
	 */
	public function add_seo_column( $columns ) {
		$columns['seobuddy'] = 'SEO Buddy';
		return $columns;
	}

	/**
	 * Render the SEO Buddy column for each post row.
	 */
	public function render_seo_column( $column, $post_id ) {
		if ( 'seobuddy' !== $column ) {
			return;
		}

		$seo_title     = get_post_meta( $post_id, 'rank_math_title', true );
		$meta_desc     = get_post_meta( $post_id, 'rank_math_description', true );
		$focus_keyword = get_post_meta( $post_id, 'rank_math_focus_keyword', true );
		$post          = get_post( $post_id );
		$title         = $seo_title ? $seo_title : $post->post_title;
		$word_count    = str_word_count( wp_strip_all_tags( $post->post_content ) );
		$has_thumb     = has_post_thumbnail( $post_id );

		$issues = 0;

		// Critical: SEO title.
		if ( strlen( $title ) < 10 ) {
			$issues++;
		}
		// Critical: Meta description.
		if ( empty( $meta_desc ) ) {
			$issues++;
		}
		// Critical: Thin content.
		if ( $word_count < 300 ) {
			$issues++;
		}
		// Warning: Focus keyword.
		if ( empty( $focus_keyword ) ) {
			$issues++;
		}
		// Warning: Featured image.
		if ( ! $has_thumb ) {
			$issues++;
		}
		// Warning: Title length.
		$title_len = strlen( $title );
		if ( $title_len < 30 || $title_len > 60 ) {
			$issues++;
		}
		// Warning: Meta desc length.
		$desc_len = strlen( $meta_desc );
		if ( $desc_len > 0 && ( $desc_len < 50 || $desc_len > 160 ) ) {
			$issues++;
		}

		$chart_svg = '<svg xmlns="http://www.w3.org/2000/svg" width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2.5" stroke-linecap="round" stroke-linejoin="round"><line x1="18" y1="20" x2="18" y2="10"/><line x1="12" y1="20" x2="12" y2="4"/><line x1="6" y1="20" x2="6" y2="14"/></svg>';

		if ( 0 === $issues ) {
			echo '<span class="seobuddy-col-badge seobuddy-col-badge--clean">' . $chart_svg . ' GTG!</span>';
		} else {
			echo '<span class="seobuddy-col-badge seobuddy-col-badge--issues">' . $chart_svg . ' ' . (int) $issues . '</span>';
		}

		if ( $focus_keyword ) {
			echo '<div class="seobuddy-col-kw">' . esc_html( $focus_keyword ) . '</div>';
		}
	}

	/**
	 * Inline CSS for the SEO Buddy column on the posts list page.
	 */
	public function seo_column_css() {
		?>
		<style>
			.column-seobuddy { width: 90px; text-align: center; vertical-align: middle !important; }
			.seobuddy-col-badge {
				display: inline-flex;
				align-items: center;
				gap: 4px;
				padding: 4px 10px;
				border-radius: 4px;
				border: 1.5px solid currentColor;
				font-size: 13px;
				font-weight: 700;
				line-height: 1;
				white-space: nowrap;
				vertical-align: middle;
			}
			.seobuddy-col-badge svg { flex-shrink: 0; }
			.seobuddy-col-badge--clean {
				color: #00742e;
				border-color: #00742e;
				background: #edfaef;
			}
			.seobuddy-col-badge--issues {
				color: #96680a;
				border-color: #c4a44a;
				background: #fef8ee;
			}
			th.column-seobuddy { text-align: center; }
			.seobuddy-col-kw {
				margin-top: 4px;
				font-size: 11px;
				color: #646970;
				text-align: center;
			}
		</style>
		<?php
	}

	/**
	 * AJAX: Poll audit status.
	 */
	public function ajax_poll_audit() {
		check_ajax_referer( 'seobuddy_nonce', '_ajax_nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => 'Unauthorized' ), 403 );
		}

		$audit_id    = isset( $_POST['audit_id'] ) ? sanitize_text_field( wp_unslash( $_POST['audit_id'] ) ) : '';
		$license_key = $this->get_license_key();

		if ( empty( $audit_id ) ) {
			wp_send_json_error( array( 'message' => 'Audit ID is required.' ) );
		}

		if ( empty( $license_key ) ) {
			wp_send_json_error( array( 'message' => 'License key not configured.' ) );
		}

		$response = wp_remote_get(
			self::API_URL . '/audits/' . urlencode( $audit_id ),
			array(
				'headers' => array(
					'X-License-Key' => $license_key,
				),
				'timeout' => 15,
			)
		);

		if ( is_wp_error( $response ) ) {
			wp_send_json_error( array( 'message' => 'Could not connect to SEO Buddy API.' ) );
		}

		$status_code = wp_remote_retrieve_response_code( $response );
		$body        = json_decode( wp_remote_retrieve_body( $response ), true );

		if ( 200 !== $status_code || empty( $body ) ) {
			wp_send_json_error( array( 'message' => 'Failed to fetch audit status.' ) );
		}

		$audit_status = isset( $body['status'] ) ? sanitize_text_field( $body['status'] ) : '';

		// Flatten report fields to top level for JS.
		$report = isset( $body['report'] ) && is_array( $body['report'] ) ? $body['report'] : array();
		$result = array(
			'status'          => $audit_status,
			'report_markdown' => isset( $report['report_markdown'] ) ? $report['report_markdown'] : '',
			'score'           => isset( $report['score'] ) ? intval( $report['score'] ) : 0,
		);

		// Save last completed audit ID.
		if ( 'complete' === $audit_status ) {
			update_option( 'seobuddy_last_audit_id', $audit_id );
		}

		wp_send_json_success( $result );
	}
}

SEOBuddy::get_instance();
